/*
 * Decompiled with CFR 0.152.
 */
package hive.keycloak.authenticator.firebase;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.FirebaseMessagingException;
import com.google.firebase.messaging.Message;
import com.google.firebase.messaging.Notification;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Time;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.RootAuthenticationSessionModel;

public class FirebaseService {
    private static final Logger logger = Logger.getLogger(FirebaseService.class);
    private static volatile boolean initialized = false;
    private static final String SESSION_NOTE_CORRECT_CODE = "firebase.correct_code";
    private static final String SESSION_NOTE_USER_ID = "firebase.user_id";
    private static final String SESSION_NOTE_EXPIRES_AT = "firebase.expires_at";
    private static final String SESSION_NOTE_STATUS = "firebase.status";
    private static final String SESSION_NOTE_VERIFICATION_TOKEN = "firebase.verification_token";
    private static final String SESSION_NOTE_REGISTRATION_USER_ID = "firebase.reg.user_id";
    private static final String SESSION_NOTE_REGISTRATION_EXPIRES_AT = "firebase.reg.expires_at";
    private static final String SESSION_NOTE_REGISTRATION_COMPLETED = "firebase.reg.completed";
    private static final String SESSION_NOTE_REGISTRATION_VERIFICATION_TOKEN = "firebase.reg.verification_token";
    private static final String STATUS_PENDING = "pending";
    private static final String STATUS_VERIFIED = "verified";
    private static final String STATUS_EXPIRED = "expired";
    private static final String STATUS_FCM_REGISTERED = "fcm_registered";
    private static final long VERIFIED_SESSION_TTL_MS = 120000L;

    public static synchronized void initialize(Map<String, String> config) {
        if (initialized) {
            return;
        }
        try {
            if (config == null) {
                throw new IllegalArgumentException("Firebase configuration cannot be null");
            }
            String serviceAccountJson = config.get("serviceAccountJson");
            if (serviceAccountJson == null || serviceAccountJson.trim().isEmpty()) {
                throw new IllegalArgumentException("Firebase Service Account JSON is required and cannot be empty");
            }
            try {
                new ObjectMapper().readTree(serviceAccountJson);
            }
            catch (JsonProcessingException e) {
                throw new IllegalArgumentException("Firebase Service Account JSON is not valid JSON format", e);
            }
            ByteArrayInputStream serviceAccount = new ByteArrayInputStream(serviceAccountJson.getBytes(StandardCharsets.UTF_8));
            FirebaseOptions options = FirebaseOptions.builder().setCredentials(GoogleCredentials.fromStream(serviceAccount)).build();
            if (FirebaseApp.getApps().isEmpty()) {
                FirebaseApp.initializeApp(options);
                logger.info((Object)"Firebase initialized successfully for FCM");
            } else {
                logger.debug((Object)"Firebase already initialized, skipping initialization");
            }
            initialized = true;
            logger.info((Object)"Firebase service initialization completed successfully");
        }
        catch (IOException e) {
            logger.error((Object)"Error initializing Firebase", (Throwable)e);
            throw new RuntimeException("Error initializing Firebase: " + e.getMessage(), e);
        }
        catch (Exception e) {
            logger.error((Object)"Unexpected error during Firebase initialization", (Throwable)e);
            throw new RuntimeException("Unexpected error initializing Firebase: " + e.getMessage(), e);
        }
    }

    public static String sendAuthNotification(KeycloakSession session, AuthenticationSessionModel authSession, String fcmToken, String sessionId, String userId, int codeDigits, int fakeCodesCount, int timeoutSeconds) throws FirebaseMessagingException {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(fcmToken, "FCM token");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        FirebaseService.validateNotEmpty(userId, "User ID");
        FirebaseService.validateRange(codeDigits, 1, 6, "Code digits");
        FirebaseService.validateRange(fakeCodesCount, 1, 5, "Fake codes count");
        FirebaseService.validateRange(timeoutSeconds, 30, 1800, "Timeout seconds");
        try {
            String correctCode = FirebaseService.generateCorrectCode(codeDigits);
            String[] fakeCodes = FirebaseService.generateFakeCodes(correctCode, fakeCodesCount, codeDigits);
            String verificationToken = FirebaseService.generateSecureToken();
            long expiryTime = Time.currentTimeMillis() + (long)timeoutSeconds * 1000L;
            authSession.setAuthNote(SESSION_NOTE_CORRECT_CODE, correctCode);
            authSession.setAuthNote(SESSION_NOTE_USER_ID, userId);
            authSession.setAuthNote(SESSION_NOTE_EXPIRES_AT, String.valueOf(expiryTime));
            authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_PENDING);
            authSession.setAuthNote(SESSION_NOTE_VERIFICATION_TOKEN, verificationToken);
            authSession.setAuthNote("firebase_session_id", sessionId);
            String rootSessionId = authSession.getParentSession().getId();
            String tabId = authSession.getTabId();
            authSession.setAuthNote("firebase_root_session_id", rootSessionId);
            authSession.setAuthNote("firebase_tab_id", tabId);
            RealmModel realm = session.getContext().getRealm();
            realm.setAttribute("firebase_session_" + sessionId, rootSessionId + ":" + tabId + ":" + realm.getClientByClientId(authSession.getClient().getClientId()).getId());
            logger.debugf("Stored session mapping for %s -> %s:%s", (Object)sessionId, (Object)rootSessionId, (Object)tabId);
            ArrayList<String> allCodes = new ArrayList<String>();
            allCodes.add(correctCode);
            allCodes.addAll(Arrays.asList(fakeCodes));
            Collections.shuffle(allCodes);
            ObjectMapper objectMapper = new ObjectMapper();
            String codesJson = objectMapper.writeValueAsString(allCodes);
            Message.Builder messageBuilder = Message.builder().setToken(fcmToken).setNotification(Notification.builder().setTitle("Login Verification").setBody("Select the code displayed on your browser").build()).putData("session_id", sessionId).putData("action", "auth_verification").putData("authenticationCodes", codesJson).putData("verification_token", verificationToken).putData("expires_at", String.valueOf(expiryTime));
            Message message = messageBuilder.build();
            String response = FirebaseMessaging.getInstance().send(message);
            logger.infof("Successfully sent auth notification to user %s, session %s, response: %s", (Object)userId, (Object)sessionId, (Object)response);
            return correctCode;
        }
        catch (JsonProcessingException e) {
            logger.error((Object)("Error serializing authentication codes to JSON for session: " + sessionId), (Throwable)e);
            throw new RuntimeException("Failed to serialize authentication codes: " + e.getMessage(), e);
        }
        catch (FirebaseMessagingException e) {
            logger.error((Object)("Firebase messaging error for session " + sessionId + ": " + e.getMessage()), (Throwable)e);
            throw e;
        }
        catch (Exception e) {
            logger.error((Object)("Unexpected error sending auth notification for session: " + sessionId), (Throwable)e);
            throw new RuntimeException("Unexpected error sending notification: " + e.getMessage(), e);
        }
    }

    public static boolean verifyAuthCode(KeycloakSession session, String sessionId, String submittedCode, String providedToken) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        FirebaseService.validateNotEmpty(submittedCode, "Submitted code");
        FirebaseService.validateNotEmpty(providedToken, "Verification token");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_session_" + sessionId);
            if (sessionMapping == null) {
                logger.warn((Object)("No session mapping found for session: " + sessionId));
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                logger.warn((Object)("Invalid session mapping format for session: " + sessionId));
                return false;
            }
            String rootSessionId = parts[0];
            String tabId = parts[1];
            String clientId = parts[2];
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, rootSessionId);
            if (rootSession == null) {
                logger.warn((Object)("Root authentication session not found for session: " + sessionId));
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(clientId), tabId);
            if (authSession == null) {
                logger.warn((Object)("Authentication session not found for session: " + sessionId));
                return false;
            }
            String correctCode = authSession.getAuthNote(SESSION_NOTE_CORRECT_CODE);
            String storedToken = authSession.getAuthNote(SESSION_NOTE_VERIFICATION_TOKEN);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_EXPIRES_AT);
            if (correctCode == null || storedToken == null || expiresAtStr == null) {
                logger.warn((Object)("Incomplete session data for session: " + sessionId));
                return false;
            }
            if (!storedToken.equals(providedToken)) {
                logger.warnf("Invalid verification token for session: %s", (Object)sessionId);
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                logger.warn((Object)("Auth session expired: " + sessionId));
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                realm.removeAttribute("firebase_session_" + sessionId);
                return false;
            }
            boolean isValid = correctCode.equals(submittedCode.trim());
            if (isValid) {
                long verifiedUntil = Time.currentTimeMillis() + 120000L;
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_VERIFIED);
                authSession.setAuthNote(SESSION_NOTE_EXPIRES_AT, String.valueOf(verifiedUntil));
                logger.infof("Auth code verified successfully for session: %s", (Object)sessionId);
            } else {
                logger.warnf("Invalid auth code submitted for session: %s", (Object)sessionId);
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                realm.removeAttribute("firebase_session_" + sessionId);
            }
            return isValid;
        }
        catch (NumberFormatException e) {
            logger.error((Object)("Invalid expiration timestamp format for session: " + sessionId), (Throwable)e);
            return false;
        }
        catch (Exception e) {
            logger.error((Object)("Error verifying auth code for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static String generateCorrectCode(int digits) {
        if (digits < 1 || digits > 6) {
            throw new IllegalArgumentException("Code digits must be between 1 and 6");
        }
        SecureRandom secureRandom = new SecureRandom();
        int min2 = (int)Math.pow(10.0, digits - 1);
        int max = (int)Math.pow(10.0, digits) - 1;
        int code = secureRandom.nextInt(max - min2 + 1) + min2;
        return String.valueOf(code);
    }

    public static String[] generateFakeCodes(String correctCode, int count, int digits) {
        if (count < 1 || count > 5) {
            throw new IllegalArgumentException("Fake codes count must be between 1 and 5");
        }
        SecureRandom secureRandom = new SecureRandom();
        HashSet<String> usedCodes = new HashSet<String>();
        usedCodes.add(correctCode);
        String[] fakeCodes = new String[count];
        int min2 = (int)Math.pow(10.0, digits - 1);
        int max = (int)Math.pow(10.0, digits) - 1;
        for (int i = 0; i < count; ++i) {
            int code;
            String fakeCode;
            while (usedCodes.contains(fakeCode = String.valueOf(code = secureRandom.nextInt(max - min2 + 1) + min2))) {
            }
            usedCodes.add(fakeCode);
            fakeCodes[i] = fakeCode;
        }
        return fakeCodes;
    }

    public static String generateSecureToken() {
        SecureRandom secureRandom = new SecureRandom();
        byte[] tokenBytes = new byte[32];
        secureRandom.nextBytes(tokenBytes);
        return Base64.getUrlEncoder().withoutPadding().encodeToString(tokenBytes);
    }

    public static boolean hasValidSession(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_session_" + sessionId);
            if (sessionMapping == null) {
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                return false;
            }
            String status = authSession.getAuthNote(SESSION_NOTE_STATUS);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_EXPIRES_AT);
            if (status == null || expiresAtStr == null) {
                return false;
            }
            if (STATUS_EXPIRED.equals(status)) {
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return false;
            }
            return STATUS_PENDING.equals(status);
        }
        catch (Exception e) {
            logger.error((Object)("Error checking session validity for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static boolean completeRegistration(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_reg_session_" + sessionId);
            if (sessionMapping == null) {
                logger.warn((Object)("No registration session mapping found for session: " + sessionId));
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                logger.warn((Object)("Invalid session mapping format for session: " + sessionId));
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                logger.warn((Object)("Root authentication session not found for session: " + sessionId));
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                logger.warn((Object)("Authentication session not found for session: " + sessionId));
                return false;
            }
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_EXPIRES_AT);
            if (expiresAtStr == null) {
                logger.warn((Object)("No expiration data found for registration session: " + sessionId));
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                logger.warn((Object)("Registration session expired: " + sessionId));
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return false;
            }
            long completedUntil = Time.currentTimeMillis() + 120000L;
            authSession.setAuthNote(SESSION_NOTE_REGISTRATION_COMPLETED, "true");
            authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_FCM_REGISTERED);
            authSession.setAuthNote(SESSION_NOTE_REGISTRATION_EXPIRES_AT, String.valueOf(completedUntil));
            logger.infof("Registration completed for session: %s", (Object)sessionId);
            return true;
        }
        catch (Exception e) {
            logger.error((Object)("Error completing registration for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static boolean hasValidRegistrationSession(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_reg_session_" + sessionId);
            if (sessionMapping == null) {
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                return false;
            }
            String userId = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_USER_ID);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_EXPIRES_AT);
            if (userId == null || expiresAtStr == null) {
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return false;
            }
            return true;
        }
        catch (Exception e) {
            logger.error((Object)("Error checking registration session validity for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static String getUserIdFromRegistrationSession(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_reg_session_" + sessionId);
            if (sessionMapping == null) {
                return null;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                return null;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                return null;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                return null;
            }
            String userId = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_USER_ID);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_EXPIRES_AT);
            if (userId == null || expiresAtStr == null) {
                return null;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return null;
            }
            return userId;
        }
        catch (Exception e) {
            logger.error((Object)("Error getting user ID from registration session: " + sessionId), (Throwable)e);
            return null;
        }
    }

    public static boolean validateRegistrationToken(KeycloakSession session, String sessionId, String providedToken) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        FirebaseService.validateNotEmpty(providedToken, "Verification token");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_reg_session_" + sessionId);
            if (sessionMapping == null) {
                logger.warn((Object)("No registration session mapping found for session: " + sessionId));
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                logger.warn((Object)("Invalid session mapping format for session: " + sessionId));
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                logger.warn((Object)("Root authentication session not found for session: " + sessionId));
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                logger.warn((Object)("Authentication session not found for session: " + sessionId));
                return false;
            }
            String storedToken = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_VERIFICATION_TOKEN);
            if (storedToken == null) {
                logger.warn((Object)("No verification token found for registration session: " + sessionId));
                return false;
            }
            boolean isValid = storedToken.equals(providedToken);
            if (!isValid) {
                logger.warnf("Invalid verification token for registration session: %s", (Object)sessionId);
            }
            return isValid;
        }
        catch (Exception e) {
            logger.error((Object)("Error validating registration token for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static boolean isRegistrationCompleted(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_reg_session_" + sessionId);
            if (sessionMapping == null) {
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                return false;
            }
            String completed = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_COMPLETED);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_REGISTRATION_EXPIRES_AT);
            if (completed == null || expiresAtStr == null) {
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return false;
            }
            return "true".equals(completed);
        }
        catch (Exception e) {
            logger.error((Object)("Error checking registration completion for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    public static boolean isSessionVerified(KeycloakSession session, String sessionId) {
        FirebaseService.validateNotNull(session, "KeycloakSession");
        FirebaseService.validateNotEmpty(sessionId, "Session ID");
        try {
            RealmModel realm = session.getContext().getRealm();
            String sessionMapping = realm.getAttribute("firebase_session_" + sessionId);
            if (sessionMapping == null) {
                return false;
            }
            String[] parts = sessionMapping.split(":");
            if (parts.length != 3) {
                return false;
            }
            RootAuthenticationSessionModel rootSession = session.authenticationSessions().getRootAuthenticationSession(realm, parts[0]);
            if (rootSession == null) {
                return false;
            }
            AuthenticationSessionModel authSession = rootSession.getAuthenticationSession(realm.getClientById(parts[2]), parts[1]);
            if (authSession == null) {
                return false;
            }
            String status = authSession.getAuthNote(SESSION_NOTE_STATUS);
            String expiresAtStr = authSession.getAuthNote(SESSION_NOTE_EXPIRES_AT);
            if (status == null || expiresAtStr == null) {
                return false;
            }
            if (!STATUS_VERIFIED.equals(status)) {
                return false;
            }
            long expiresAt = Long.parseLong(expiresAtStr);
            if (Time.currentTimeMillis() > expiresAt) {
                authSession.setAuthNote(SESSION_NOTE_STATUS, STATUS_EXPIRED);
                return false;
            }
            return true;
        }
        catch (Exception e) {
            logger.error((Object)("Error checking session verification status for session: " + sessionId), (Throwable)e);
            return false;
        }
    }

    private static void validateNotNull(Object value, String fieldName) {
        if (value == null) {
            throw new IllegalArgumentException(fieldName + " cannot be null");
        }
    }

    private static void validateNotEmpty(String value, String fieldName) {
        if (value == null || value.trim().isEmpty()) {
            throw new IllegalArgumentException(fieldName + " cannot be null or empty");
        }
    }

    private static void validateRange(int value, int min2, int max, String fieldName) {
        if (value < min2 || value > max) {
            throw new IllegalArgumentException(fieldName + " must be between " + min2 + " and " + max + ", got: " + value);
        }
    }
}

